# fails_open_integrity.py
import hashlib
import json
import os
import platform
import subprocess
import sys
from pathlib import Path
from datetime import datetime
from typing import Optional

def get_stable_machine_fingerprint() -> str:
    """Get a stable machine fingerprint for cross-platform hardware identification"""
    try:
        system = platform.system().lower()
        
        if system == "windows":
            return _get_windows_machine_id()
        elif system == "linux":
            return _get_linux_machine_id()
        elif system == "darwin":  # macOS
            return _get_macos_machine_id()
        else:
            return _get_fallback_machine_id()
    except Exception:
        # Fallback to hostname if all hardware detection fails
        return _get_fallback_machine_id()

def _get_windows_machine_id() -> str:
    """Get Windows machine ID using volume serial number"""
    try:
        # Try to get volume serial number (most stable)
        result = subprocess.run(
            ['wmic', 'logicaldisk', 'get', 'volumeserialnumber'],
            capture_output=True, text=True, timeout=5
        )
        if result.returncode == 0:
            lines = result.stdout.strip().split('\n')
            for line in lines:
                serial = line.strip()
                if serial and serial != 'VolumeSerialNumber':
                    return hashlib.sha256(serial.encode()).hexdigest()[:16]
    except Exception:
        pass
    
    # Fallback to UUID if wmic fails
    try:
        result = subprocess.run(
            ['wmic', 'csproduct', 'get', 'uuid'],
            capture_output=True, text=True, timeout=5
        )
        if result.returncode == 0:
            lines = result.stdout.strip().split('\n')
            for line in lines:
                uuid = line.strip()
                if uuid and uuid != 'UUID':
                    return hashlib.sha256(uuid.encode()).hexdigest()[:16]
    except Exception:
        pass
    
    return _get_fallback_machine_id()

def _get_linux_machine_id() -> str:
    """Get Linux machine ID using DMI or block device info"""
    # Try DMI UUID first (most stable)
    try:
        with open('/sys/class/dmi/id/product_uuid', 'r') as f:
            uuid = f.read().strip()
            if uuid:
                return hashlib.sha256(uuid.encode()).hexdigest()[:16]
    except Exception:
        pass
    
    # Try block device serial numbers
    try:
        result = subprocess.run(
            ['lsblk', '-d', '-n', '-o', 'serial'],
            capture_output=True, text=True, timeout=5
        )
        if result.returncode == 0:
            lines = result.stdout.strip().split('\n')
            for line in lines:
                serial = line.strip()
                if serial:
                    return hashlib.sha256(serial.encode()).hexdigest()[:16]
    except Exception:
        pass
    
    # Try machine-id file
    try:
        with open('/etc/machine-id', 'r') as f:
            machine_id = f.read().strip()
            if machine_id:
                return hashlib.sha256(machine_id.encode()).hexdigest()[:16]
    except Exception:
        pass
    
    return _get_fallback_machine_id()

def _get_macos_machine_id() -> str:
    """Get macOS machine ID using IOPlatformExpertDevice"""
    try:
        result = subprocess.run(
            ['ioreg', '-d2', '-c', 'IOPlatformExpertDevice'],
            capture_output=True, text=True, timeout=5
        )
        if result.returncode == 0:
            # Look for IOPlatformUUID
            for line in result.stdout.split('\n'):
                if 'IOPlatformUUID' in line:
                    # Extract UUID from the line
                    parts = line.split('"')
                    if len(parts) >= 2:
                        uuid = parts[1]
                        return hashlib.sha256(uuid.encode()).hexdigest()[:16]
    except Exception:
        pass
    
    # Fallback to system profiler
    try:
        result = subprocess.run(
            ['system_profiler', 'SPHardwareDataType'],
            capture_output=True, text=True, timeout=5
        )
        if result.returncode == 0:
            for line in result.stdout.split('\n'):
                if 'Serial Number' in line:
                    serial = line.split(':')[-1].strip()
                    if serial:
                        return hashlib.sha256(serial.encode()).hexdigest()[:16]
    except Exception:
        pass
    
    return _get_fallback_machine_id()

def _get_fallback_machine_id() -> str:
    """Fallback to hostname-based identification"""
    try:
        hostname = platform.node()
        if hostname:
            return hashlib.sha256(hostname.encode()).hexdigest()[:16]
    except Exception:
        pass
    
    # Final fallback - generate a consistent hash
    return hashlib.sha256(b"fallback_machine_id").hexdigest()[:16]

class FailsOpenIntegrity:
    """Installation integrity checker with fail-open behavior"""
    
    def __init__(self, app_path: str):
        self.app_path = Path(app_path)
        self.integrity_file = self.app_path / ".aimms_install"
        self.integrity_failed = False
    
    def check_integrity(self) -> bool:
        """Check installation integrity with fail-open behavior"""
        try:
            # If no integrity file exists, create one (first run)
            if not self.integrity_file.exists():
                self._create_integrity_marker()
                return True
            
            # Verify integrity with fail-open approach
            return self._verify_integrity_fails_open()
            
        except Exception as e:
            # ✅ FAIL OPEN: Log error but don't block application
            self._log_integrity_warning(f"Integrity check failed: {e}")
            self.integrity_failed = True
            return True  # Always return True to allow app to run
    
    def _create_integrity_marker(self):
        """Create installation marker (may fail silently)"""
        try:
            marker_data = {
                "install_time": datetime.now().isoformat(),
                "machine_fingerprint": get_stable_machine_fingerprint(),
                "app_version": "1.0",
                "install_path_hash": hashlib.sha256(str(self.app_path).encode()).hexdigest()[:8]
            }
            
            with open(self.integrity_file, 'w') as f:
                json.dump(marker_data, f, indent=2)
                
        except Exception as e:
            # ✅ FAIL OPEN: Silent failure
            self._log_integrity_warning(f"Could not create integrity marker: {e}")
    
    def _verify_integrity_fails_open(self) -> bool:
        """Verify installation with fail-open behavior"""
        try:
            with open(self.integrity_file, 'r') as f:
                marker_data = json.load(f)
            
            current_machine = get_stable_machine_fingerprint()
            stored_machine = marker_data.get("machine_fingerprint")
            
            # Check machine fingerprint (forgiving)
            if current_machine != stored_machine:
                if self._is_reasonable_machine_change(stored_machine, current_machine):
                    # Update stored machine ID for legitimate changes
                    marker_data["machine_fingerprint"] = current_machine
                    try:
                        with open(self.integrity_file, 'w') as f:
                            json.dump(marker_data, f, indent=2)
                    except Exception:
                        pass  # Silent failure
                    return True
                else:
                    # ✅ FAIL OPEN: Log warning but allow app to run
                    self._log_integrity_warning(f"Machine fingerprint mismatch: {stored_machine} -> {current_machine}")
                    return True  # Still return True to allow app to run
            
            return True
            
        except Exception as e:
            # ✅ FAIL OPEN: Log error but don't block
            self._log_integrity_warning(f"Integrity verification failed: {e}")
            return True  # Always return True
    
    def _is_reasonable_machine_change(self, old_machine: str, new_machine: str) -> bool:
        """Check if machine change is reasonable (forgiving)"""
        # Allow changes if they share common prefixes
        if old_machine[:8] == new_machine[:8]:
            return True
        
        # Allow fallback machine IDs
        if "mac_fallback" in old_machine and "mac_fallback" in new_machine:
            return True
        
        # Allow hostname-based changes
        if old_machine.isalnum() and new_machine.isalnum():
            return True
        
        return False
    
    def _log_integrity_warning(self, message: str):
        """Log integrity warning (optional, non-critical)"""
        try:
            log_file = Path.home() / ".aimms_integrity_log"
            timestamp = datetime.now().isoformat()
            with open(log_file, 'a') as f:
                f.write(f"{timestamp}: {message}\n")
        except Exception:
            pass  # Silent failure - logging is optional
    
    def is_integrity_failed(self) -> bool:
        """Check if integrity failed (for feature gating, not app blocking)"""
        return self.integrity_failed